#include "stdlib.h"
#include "stdbool.h"
#include "string.h"

#include "usb_otg.h"
#include "usb_dev.h"
#include "usb_dev_descr.h"
#include "usb_cdc_vcp.h"

#define EP_DATA_IN (EP1_INOUT | EP1_ADDRESS)
#define EP_DATA_OUT (EP2_INOUT | EP2_ADDRESS)
#define EP_STATUS_IN (EP3_INOUT | EP3_ADDRESS)

#if defined ( __ICCARM__ ) /*!< IAR Compiler */
  #pragma data_alignment = 4   
#endif

uint8_t cfg_dscr [0x43] = 
{
  sizeof(CONFIGURATION_DSCR_TYPE),
  USB_DESC_TYPE_CONFIGURATION,
  LOBYTE(sizeof(cfg_dscr)),
  HIBYTE(sizeof(cfg_dscr)),
  0x02, //bNumInterfaces
  0x01, //bConfigurationValue,
  0x00, //iConfiguration
  0xc0, // ???
  0x32, //max 100 mA

  sizeof(INTERFACE_DSCR_TYPE),
  USB_DESC_TYPE_INTERFACE,
  0x00, //bInterfaceNumber
  0x00, //bAlternateSetting
  0x01, //bNumEndpoints
  0x02, //bInterfaceClass
  0x02, //bInterfaceSubClass
  0x01, //bInterfaceProtocol
  0x00, //iInterface
  //header func descr
  0x05, // bLength (EP Descr size)
  0x24, // bDescriptorType (CS_INTERFACE)
  0x00, // bDescriptorSubType (Header func Descr)
  0x10, 0x01, //bcd CDC
  // Call Management Function
  0x05, // bFunctionLength
  0x24, // bDescrType
  0x01, // bDescrSubType
  0x00, // bmCapabilities
  0x01, // bDataInterface
  // ACM functional Descr
  0x04, // bFunctionLength
  0x24, // bDescrType (CS_INTERFACE)
  0x02, // bDescrSubType (ACM)
  0x00,//0x06, // bmCapabilities
  // Union Func Descr
  0x05, // bFunctionLength
  0x24, // bDescriptorType (CS_INTERFACE)
  0x06, // bDescriptorSubtype (Union func desc)
  0x00, // bMasterInterface (Communication class interface)
  0x01, // bSlaveInterface0 (Data Class Interface)
  // EP 3 Descr
  0x07, // bLength (Endpoint Descriptor size) 
  0x05, // bDescriptorType (Endpoint) 
  EP_STATUS_IN, // bEndpointAddress (IN | 3) 
  EP3_ATTRIB, // bmAttributes (Interrupt)
  LOBYTE(EP3_MAXPACKSIZE),
  HIBYTE(EP3_MAXPACKSIZE), //  wMaxPacketSize
  EP3_INTERVAL, // bInterval
  // Data class interface Descriptor
  0x09, // bLength
  0x04, // bDescriptorType (Interface)
  0x01, // bInterfaceNumber
  0x00, // bAlternateSetting
  0x02, // bNumEndpoints (Two endpoints used)
  0x0a, // bInterfaceClass (CDC)
  0x00, // bInterfaceSubClass
  0x00, // bInterfaceProtocol
  0x00, // iInterface
  // EP 1 Descr
  0x07, // bLength (Endpoint Descriptor size)        
  0x05, // bDescriptorType (Endpoint)                
  EP_DATA_IN, // bEndpointAddress (IN | 1)                 
  EP1_ATTRIB, // bmAttributes (Bulk)                       
  LOBYTE(EP1_MAXPACKSIZE),
  HIBYTE(EP1_MAXPACKSIZE), //  wMaxPacketSize               
  EP1_INTERVAL, // bInterval                          
  // EP 2 Descr
  0x07, // bLength (Endpoint Descriptor size)        
  0x05, // bDescriptorType (Endpoint)                
  EP_DATA_OUT, // bEndpointAddress (OUT | 2)                 
  EP2_ATTRIB, // bmAttributes (Bulk)                       
  LOBYTE(EP2_MAXPACKSIZE),
  HIBYTE(EP2_MAXPACKSIZE), //  wMaxPacketSize                  
  EP2_INTERVAL // bInterval 
};

//////////////////////////////////////////////////////
uint32_t USBDev_Class_Init(uint8_t config_num);
uint32_t USBDev_Class_DeInit(uint8_t config_num);
uint32_t USBDev_Class_DataOut(USBDev_EP *ep);
uint32_t USBDev_Class_DataIn(USBDev_EP *ep);
uint32_t USBDev_Class_Setup(USBDev_Setup_TypeDef *pUSB_SETUP_PACKET);
uint32_t USBDev_Class_SOF(void);
uint32_t USBDev_Class_EP0_RxReady(void);
uint32_t USBDev_Class_EP0_TxSent(void);
uint8_t *USBDev_Class_GetConfigDscr(USBDEV_SPEED dev_speed, uint32_t *len);
//////////////////////////////////////////////////////
extern uint32_t VCP_CDC_DataTransmitted(void);
extern uint32_t VCP_CDC_DataReceived(void);

//////////////////////////////////////////////////////
// buffer settings for VCP
#define BUF_SIZE 128
#define BUF_MASK (BUF_SIZE - 1)

uint8_t buf_rx[BUF_SIZE];
uint8_t buf_tx[BUF_SIZE];

uint32_t wrptr_rxbuf = 0;
uint32_t rdptr_rxbuf = 0;
uint32_t fill_rxbuf = 0;

uint32_t wrptr_txbuf = 0;
uint32_t rdptr_txbuf = 0;
uint32_t fill_txbuf = 0;
//////////////////////////////////////////////////////

uint8_t cdc_tx_now = 0;
uint8_t line_coding_struct[7]; //

uint32_t Handle_TX(uint8_t epnum);
uint32_t Handle_RX(uint8_t epnum);

//////////////////////////////////////////////////////
//   
//////////////////////////////////////////////////////
uint32_t USBDev_Class_Init(uint8_t config_num)
{
  //in DATA EP1
  USBDev_EP_Open(EP_DATA_IN,
                 64,
                 2); //IN,maxpack=64,Bulk
  //out DATA EP2
  USBDev_EP_Open(EP_DATA_OUT,
                 64,
                 2); //OUT,maxpack=64,Bulk
  //command IN EP
  USBDev_EP_Open(EP_STATUS_IN,
                 64,
                 3); //IN,maxpack=64,Int
  
  //enable receiving data
  USBDev_EPSetResponse(EP_CLRNAK, EP_DATA_OUT);
  
  return 0;
}

uint32_t USBDev_Class_DeInit(uint8_t config_num)
{
  //in DATA EP1
  USBDev_EP_Close(EP_DATA_IN);
  //out DATA EP2
  USBDev_EP_Close(EP_DATA_OUT);
  //command IN EP
  USBDev_EP_Close(EP_STATUS_IN);
  
  return 0;
}

uint32_t USBDev_Class_DataOut(USBDev_EP *ep)
{
  if(ep->epnum & EP_DATA_OUT & 0x7F)
    return Handle_RX(EP_DATA_OUT & 0x7F);
  
  return 0;  
}

uint32_t USBDev_Class_DataIn(USBDev_EP *ep)
{
  if(ep->epnum & EP_DATA_IN & 0x7F)
    Handle_TX(EP_DATA_IN & 0x7F);
  
  return 0;  
}

uint32_t USBDev_Class_Setup(USBDev_Setup_TypeDef *pUSB_SETUP_PACKET)
{
  switch(pUSB_SETUP_PACKET->bmRequestType & 0x60)
  {
  //    
  case 0x20: //CLASS_SPECIFIC_REQ
    switch (pUSB_SETUP_PACKET->bRequest)
    {
    case SET_LINE_CODING: 
      if(pUSB_SETUP_PACKET->wLength == 7)
      {
        //  ,   
        USBDev_EPPrepareRx(&line_coding_struct[0], 7, 0);
        
        /*pUSB_SETUP_PACKET->xfer_buff = (uint8_t*)&line_coding_struct[0];
        pUSB_SETUP_PACKET->xfer_count = 0;
        pUSB_SETUP_PACKET->xfer_len = pUSB_SETUP_PACKET->wLength;
        
        USBDev_ReadPacket((uint8_t*)&line_coding_struct[0],
                          7,
                          0);*/
        //USBDev_CEPSendResponse(CEP_ZEROLEN);
      }
      else
      {
        USBDev_CEPSendResponse(CEP_STALL);
      }
      break;
    case GET_LINE_CODING: 
      if(pUSB_SETUP_PACKET->wLength == 7)
      {        
        /*USBDev_CEPSendData((uint8_t*)&line_coding_struct[0],
                           7);*/
        
        USBDev_CEPSendResponse(CEP_STALL);
        
        
      }
      else
      {
        USBDev_CEPSendResponse(CEP_STALL);
      }
      break;
      
    case SET_CONTROL_LINE_STATE: 
      //     
      //     wValue
      USBDev_CEPSendResponse(CEP_ZEROLEN);
      break;
      
    default:
      USBDev_CEPSendResponse(CEP_STALL);
      break;
    }
    break;
  
  default:
    break;
    
  }
  
  if(pUSB_SETUP_PACKET->wLength == 0)
  {
    if(pUSB_SETUP_PACKET->bmRequestType & 0x80) //device to host
    {
      
    }
    else //host to device
    {
      USBDev_CEPSendResponse(CEP_ZEROLEN); 
    }
  }
  
  return 0;
}

uint32_t USBDev_Class_SOF(void)
{
  if((cdc_tx_now == 0) && (USBDev_ReadState() == USBDEV_CONFIGURED))
    Handle_TX(EP_DATA_IN & 0x7F);
  
  //if rx buffer have some data
  if(VCP_IsRxBufEmpty() == 0)
    VCP_CDC_DataReceived();    
    
  return 0;  
}

uint32_t USBDev_Class_EP0_RxReady(void)
{
  return 0;  
}

uint32_t USBDev_Class_EP0_TxSent(void)
{
  return 0;  
}

uint8_t *USBDev_Class_GetConfigDscr(USBDEV_SPEED dev_speed, uint32_t *len)
{
  uint8_t *pcfgdscr;
  if(dev_speed == USBDEV_SPEED_FULL)
  {
    *len = sizeof(cfg_dscr);
    pcfgdscr = &cfg_dscr[0];
  }
  else //HIGH
  {
    *len = sizeof(cfg_dscr);
    pcfgdscr = &cfg_dscr[0];
  }
  return pcfgdscr;
}

////////////////////////////////////////////////////////////////////////////////
uint32_t Handle_RX(uint8_t epnum)
{
  uint32_t diff;
  //   ,    
  uint32_t rx_bytes_cnt = USBDev_EP_AvailCnt(epnum); 
  //       
  //  
  if((BUF_SIZE - fill_rxbuf) >= rx_bytes_cnt)
  {
    //       
    if((BUF_SIZE - wrptr_rxbuf) >= rx_bytes_cnt)
    {      
      USBDev_ReadPacket((uint8_t*)&buf_rx[wrptr_rxbuf],
                         rx_bytes_cnt,
                         epnum);
      
      //    
      fill_rxbuf += rx_bytes_cnt;
      //  
      wrptr_rxbuf = (wrptr_rxbuf + rx_bytes_cnt) & BUF_MASK;
    }
    //      ,     
    else 
    {
      diff = BUF_SIZE - wrptr_rxbuf;
      //     
      
      USBDev_ReadPacket((uint8_t*)&buf_rx[wrptr_rxbuf],
                         diff,
                         epnum);
            
      // 
      //    
      fill_rxbuf += diff;
      //  
      wrptr_rxbuf = (wrptr_rxbuf + diff) & BUF_MASK;
      
      if(fill_rxbuf == BUF_SIZE)
        return 1;
      
      USBDev_ReadPacket((uint8_t*)&buf_rx[0],
                         rx_bytes_cnt - diff,
                         epnum);
      
      fill_rxbuf += rx_bytes_cnt - diff;
      wrptr_rxbuf = rx_bytes_cnt - diff;
    }
    return 0;  
  }
  else //   ,       
  {
    return 1;
  }
}

uint8_t need_zlp = 0;
uint32_t Handle_TX(uint8_t epnum)
{
  uint32_t data_len;
  //    
  if(fill_txbuf != 0)
  {
    //  
    cdc_tx_now = 1;
    
    if(rdptr_txbuf >= wrptr_txbuf)
    {
      if((BUF_SIZE - rdptr_txbuf) >= 64)
      {
        data_len = 64;
        need_zlp = 1;
      }
      else
      {
        data_len = BUF_SIZE - rdptr_txbuf;
        need_zlp = 0;
      }
      
      USBDev_WritePacket((uint8_t*)&buf_tx[rdptr_txbuf],
                         data_len,
                         epnum);
      
      rdptr_txbuf = (rdptr_txbuf + data_len) & BUF_MASK;     
      fill_txbuf -= data_len;    
    }
    else //rdptr_txbuf < wrptr_txbuf
    {
      if(fill_txbuf >= 64)
      {
        data_len = 64;
        need_zlp = 1;
      }
      else
      {
        data_len = fill_txbuf;
        need_zlp = 0;
      }
      
      USBDev_WritePacket((uint8_t*)&buf_tx[rdptr_txbuf],
                         data_len,
                         epnum);
      
      rdptr_txbuf += data_len;
      fill_txbuf -= data_len;
    }    
  }
  //   ,     
  else 
  {
    if(cdc_tx_now)
    {
      //     
      if(need_zlp)
        USBDev_EPSetResponse(EP_ZEROLEN, epnum);
      
      //   
      cdc_tx_now = 0;  
    }
    else
    {
      //CALL DATA_SEND_COMPLETE_FUNC  
      VCP_CDC_DataTransmitted();
    }
  }
  
  return 0;  
}

uint32_t VCP_PutChar(uint8_t byte)
{
  if (VCP_IsTxBufFull())
    return 1;

  buf_tx[wrptr_txbuf] = byte;
  wrptr_txbuf = (wrptr_txbuf + 1) & BUF_MASK;
  fill_txbuf++;

  return 0;
}

uint8_t VCP_GetChar(void)
{  
  uint8_t byte;
  
  byte = buf_rx[rdptr_rxbuf];
  rdptr_rxbuf = (rdptr_rxbuf + 1) & BUF_MASK;
  fill_rxbuf--;
  
  return byte;
}

uint32_t VCP_IsRxBufEmpty(void)
{
  if(fill_rxbuf == 0) return 1;
  else                return 0;  
}

uint32_t VCP_IsTxBufFull(void)
{
  return (fill_txbuf == BUF_SIZE);
}
